﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using System.Threading;
using System.Timers;


/*  ThreadPool class  */
namespace Lessons
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent(); 
        }

        const int COMMAND_COUNT = 35;

        EventWaitHandle eventWaitHandle;
        string[] lines;
        int commandsLeft = 0;  

        /* it is called by a Command object and provides feedback about the operation.
           It first checks if Control.Invoked() is required and if yes then it recalls itself
           using Invoke(), in order to be executed in a synchronized manner */
        void JobInfoCallBack(JobInfo info)
        { 
            if (progressBar1.InvokeRequired)
                progressBar1.Invoke(new JobInfoDelegate(JobInfoCallBack), info);
            else
            {
                Command cmd = (Command)info.UserState;                

                switch (info.Status)
                {
                    case JobStatus.Working: 
                        lines[cmd.ID] = "ID: " + cmd.ID.ToString() + ", working: " + cmd.DonePercent.ToString() + "%, Total size: " + cmd.DataSize.ToString() + " (" + cmd.RemainSize.ToString() + " left)";
                        textBox1.Lines = lines;
                        break;
                    case JobStatus.Completed:
                        progressBar1.Value = ((COMMAND_COUNT - commandsLeft) * 100) / COMMAND_COUNT;
                        lines[cmd.ID] = "ID: " + cmd.ID.ToString() + "   DONE";
                        textBox1.Lines = lines;
                        commandsLeft--;
                        break;
                    case JobStatus.Aborted:
                        commandsLeft--;          

                        break;
                    case JobStatus.Error:
                        lines[cmd.ID] = "ID: " + cmd.ID.ToString() + "   ERROR: " + info.Message;
                        textBox1.Lines = lines;
                        commandsLeft--;
                        break;
                }

                if (commandsLeft <= 0)
                    progressBar1.Value = 0;
            }
        }

        /* the call back for the ThreadPool.QueueUserWorkItem() */
        void QueueUserWorkItem_CallBack(object state)
        {
            ((Command)state).Execute(JobInfoCallBack);
        }

        /* the call back for the ThreadPool.RegisterWaitForSingleObject() */
        void RegisterWaitForSingleObject_CallBack(object state, bool timedOut)
        {
            ((Command)state).Execute(JobInfoCallBack);
        }

        /* starts a ThreadPool.QueueUserWorkItem() operation */
        private void btnStart_Click(object sender, EventArgs e)
        {
            if (commandsLeft == 0)
            {
                commandsLeft = COMMAND_COUNT;

                textBox1.Text = "";
                lines = new string[COMMAND_COUNT];

                for (int i = 0; i < COMMAND_COUNT; i++)
                    ThreadPool.QueueUserWorkItem(QueueUserWorkItem_CallBack, new Command(i));
 
            }
        }

        /* starts a ThreadPool.RegisterWaitForSingleObject() operation */
        private void btnStartWaiting_Click(object sender, EventArgs e)
        {
            if (commandsLeft == 0)
            {
                eventWaitHandle = new ManualResetEvent(false);
                commandsLeft = COMMAND_COUNT;

                textBox1.Text = "";
                lines = new string[COMMAND_COUNT];

                for (int i = 0; i < COMMAND_COUNT; i++)
                {
                    ThreadPool.RegisterWaitForSingleObject(
                        eventWaitHandle,                        // WaitHandle waitObject
                        RegisterWaitForSingleObject_CallBack,   // WaitOrTimerCallback callBack
                        new Command(i),                         // Object state      
                        -1,                                     // int millisecondsTimeOutInterval 
                        true                                    // bool executeOnlyOnce
                        );                    
                }

                eventWaitHandle.Set();
            }
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = commandsLeft > 0;

            if (e.Cancel)
                MessageBox.Show("Please wait! Thread pool threads are still executed...");
        }


    }
}



namespace Lessons
{

    public enum JobStatus
    {
        Working,
        Completed,
        Aborted,
        Error
    }

    /* conveys information regarding a queued task */
    public class JobInfo
    {
        private JobStatus status;
        private object userState;
        private string message;

        public JobInfo(JobStatus Status, object UserState, string Message)
        {
            status = Status;
            userState = UserState;
            message = Message;
        }

        public JobStatus Status { get { return status; } }
        public object UserState { get { return userState; } }
        public string Message { get { return message; } }
    }

    public delegate void JobInfoDelegate(JobInfo info);


    /* Represents a job queued for execution using a thread pool thread.
       The Execute() method is called from inside the thread context 
       and sends feed back information using the passed delegate value */
    class Command
    {
        static Random random = new Random();

        private int id;
        private int dataSize = random.Next(30, 50);
        private int remainSize = 0;
        private int donePercent = 0;

        public Command(int ID)
        {
            id = ID;
        }

        public void Execute(JobInfoDelegate JobInfoCallBack)
        {
            remainSize = dataSize;
            donePercent = 0;

            try
            {
                while (remainSize > 0)
                {
                    remainSize--;
                    donePercent = ((dataSize - remainSize) * 100) / dataSize;

                    JobInfoCallBack(new JobInfo(JobStatus.Working, this, ""));
                    Thread.Sleep(150);
                }

                JobInfoCallBack(new JobInfo(JobStatus.Completed, this, ""));
            }
            catch (Exception ex)
            {
                JobInfoCallBack(new JobInfo(JobStatus.Error, this, ex.GetType().FullName + ": " + ex.Message));
            }
        }

        public int ID { get { return id; } }
        public int DataSize { get { return dataSize; } }
        public int RemainSize { get { return remainSize; } }
        public int DonePercent { get { return donePercent; } }
    }

 
}